/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.commons.pipe.connector.protocol;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.net.InetSocketAddress;
import java.net.Socket;
import java.net.SocketException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.zip.CRC32;
import org.apache.iotdb.common.rpc.thrift.TEndPoint;
import org.apache.iotdb.common.rpc.thrift.TSStatus;
import org.apache.iotdb.commons.pipe.config.PipeConfig;
import org.apache.iotdb.commons.pipe.connector.payload.airgap.AirGapELanguageConstant;
import org.apache.iotdb.commons.pipe.connector.payload.airgap.AirGapOneByteResponse;
import org.apache.iotdb.commons.pipe.connector.protocol.IoTDBConnector;
import org.apache.iotdb.pipe.api.annotation.TableModel;
import org.apache.iotdb.pipe.api.annotation.TreeModel;
import org.apache.iotdb.pipe.api.customizer.configuration.PipeConnectorRuntimeConfiguration;
import org.apache.iotdb.pipe.api.customizer.parameter.PipeParameters;
import org.apache.iotdb.pipe.api.exception.PipeConnectionException;
import org.apache.iotdb.pipe.api.exception.PipeException;
import org.apache.iotdb.rpc.TSStatusCode;
import org.apache.tsfile.utils.BytesUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@TreeModel
@TableModel
public abstract class IoTDBAirGapConnector
extends IoTDBConnector {
    private static final Logger LOGGER = LoggerFactory.getLogger(IoTDBAirGapConnector.class);
    protected static final PipeConfig PIPE_CONFIG = PipeConfig.getInstance();
    protected final List<AirGapSocket> sockets = new ArrayList<AirGapSocket>();
    protected final List<Boolean> isSocketAlive = new ArrayList<Boolean>();
    private LoadBalancer loadBalancer;
    private long currentClientIndex = 0L;
    private int handshakeTimeoutMs;
    private boolean eLanguageEnable;
    protected boolean supportModsIfIsDataNodeReceiver = true;
    private final Map<TEndPoint, Long> failLogTimes = new HashMap<TEndPoint, Long>();

    @Override
    public void customize(PipeParameters parameters, PipeConnectorRuntimeConfiguration configuration) throws Exception {
        super.customize(parameters, configuration);
        if (this.isTabletBatchModeEnabled) {
            LOGGER.warn("Batch mode is enabled by the given parameters. IoTDBAirGapConnector does not support batch mode. Disable batch mode.");
        }
        for (int i = 0; i < this.nodeUrls.size(); ++i) {
            this.isSocketAlive.add(false);
            this.sockets.add(null);
        }
        switch (this.loadBalanceStrategy) {
            case "round-robin": {
                this.loadBalancer = new RoundRobinLoadBalancer();
                break;
            }
            case "random": {
                this.loadBalancer = new RandomLoadBalancer();
                break;
            }
            case "priority": {
                this.loadBalancer = new PriorityLoadBalancer();
                break;
            }
            default: {
                LOGGER.warn("Unknown load balance strategy: {}, use round-robin strategy instead.", (Object)this.loadBalanceStrategy);
                this.loadBalancer = new RoundRobinLoadBalancer();
            }
        }
        this.handshakeTimeoutMs = parameters.getIntOrDefault(Arrays.asList("connector.air-gap.handshake-timeout-ms", "sink.air-gap.handshake-timeout-ms"), 5000);
        LOGGER.info("IoTDBAirGapConnector is customized with handshakeTimeoutMs: {}.", (Object)this.handshakeTimeoutMs);
        this.eLanguageEnable = parameters.getBooleanOrDefault(Arrays.asList("connector.air-gap.e-language.enable", "sink.air-gap.e-language.enable"), false);
        LOGGER.info("IoTDBAirGapConnector is customized with eLanguageEnable: {}.", (Object)this.eLanguageEnable);
    }

    public void handshake() throws Exception {
        int i;
        for (i = 0; i < this.sockets.size(); ++i) {
            if (Boolean.TRUE.equals(this.isSocketAlive.get(i))) continue;
            String ip = ((TEndPoint)this.nodeUrls.get(i)).getIp();
            int port = ((TEndPoint)this.nodeUrls.get(i)).getPort();
            if (this.sockets.get(i) != null) {
                try {
                    ((AirGapSocket)this.sockets.set(i, null)).close();
                }
                catch (Exception e) {
                    LOGGER.warn("Failed to close socket with target server ip: {}, port: {}, because: {}. Ignore it.", new Object[]{ip, port, e.getMessage()});
                }
            }
            AirGapSocket socket = new AirGapSocket(ip, port);
            try {
                socket.connect(new InetSocketAddress(ip, port), this.handshakeTimeoutMs);
                socket.setKeepAlive(true);
                this.sockets.set(i, socket);
                LOGGER.info("Successfully connected to target server ip: {}, port: {}.", (Object)ip, (Object)port);
                this.failLogTimes.remove(this.nodeUrls.get(i));
            }
            catch (Exception e) {
                TEndPoint endPoint = (TEndPoint)this.nodeUrls.get(i);
                long currentTimeMillis = System.currentTimeMillis();
                Long lastFailLogTime = this.failLogTimes.get(endPoint);
                if (lastFailLogTime != null && currentTimeMillis - lastFailLogTime <= 60000L) continue;
                this.failLogTimes.put(endPoint, currentTimeMillis);
                LOGGER.warn("Failed to connect to target server ip: {}, port: {}, because: {}. Ignore it.", new Object[]{ip, port, e.getMessage()});
                continue;
            }
            try {
                this.sendHandshakeReq(socket);
                this.isSocketAlive.set(i, true);
                continue;
            }
            catch (Exception e) {
                LOGGER.warn("Handshake error occurs. It may be caused by an error on the receiving end. Ignore it.", (Throwable)e);
            }
        }
        for (i = 0; i < this.sockets.size(); ++i) {
            if (!Boolean.TRUE.equals(this.isSocketAlive.get(i))) continue;
            return;
        }
        throw new PipeConnectionException(String.format("All target servers %s are not available.", this.nodeUrls));
    }

    protected void sendHandshakeReq(AirGapSocket socket) throws IOException {
        socket.setSoTimeout(this.handshakeTimeoutMs);
        if (!this.send(socket, this.generateHandShakeV2Payload())) {
            this.supportModsIfIsDataNodeReceiver = false;
            if (!this.send(socket, this.generateHandShakeV1Payload())) {
                throw new PipeConnectionException("Handshake error with target server, socket: " + socket);
            }
        } else {
            this.supportModsIfIsDataNodeReceiver = true;
        }
        socket.setSoTimeout(PIPE_CONFIG.getPipeConnectorTransferTimeoutMs());
        LOGGER.info("Handshake success. Socket: {}", (Object)socket);
    }

    protected abstract byte[] generateHandShakeV1Payload() throws IOException;

    protected abstract byte[] generateHandShakeV2Payload() throws IOException;

    public void heartbeat() {
        try {
            this.handshake();
        }
        catch (Exception e) {
            LOGGER.warn("Failed to reconnect to target server, because: {}. Try to reconnect later.", (Object)e.getMessage(), (Object)e);
        }
    }

    protected void transferFilePieces(String pipeName, long creationTime, File file, AirGapSocket socket, boolean isMultiFile) throws PipeException, IOException {
        int readFileBufferSize = PipeConfig.getInstance().getPipeConnectorReadFileBufferSize();
        byte[] readBuffer = new byte[readFileBufferSize];
        long position = 0L;
        try (RandomAccessFile reader = new RandomAccessFile(file, "r");){
            int readLength;
            while ((readLength = reader.read(readBuffer)) != -1) {
                byte[] payload = readLength == readFileBufferSize ? readBuffer : Arrays.copyOfRange(readBuffer, 0, readLength);
                if (!this.send(pipeName, creationTime, socket, isMultiFile ? this.getTransferMultiFilePieceBytes(file.getName(), position, payload) : this.getTransferSingleFilePieceBytes(file.getName(), position, payload))) {
                    String errorMessage = String.format("Transfer file %s error. Socket %s.", file, socket);
                    if (this.mayNeedHandshakeWhenFail()) {
                        this.sendHandshakeReq(socket);
                    }
                    this.receiverStatusHandler.handle(new TSStatus(TSStatusCode.PIPE_RECEIVER_USER_CONFLICT_EXCEPTION.getStatusCode()).setMessage(errorMessage), errorMessage, file.toString());
                    continue;
                }
                position += (long)readLength;
            }
        }
    }

    protected abstract boolean mayNeedHandshakeWhenFail();

    protected abstract byte[] getTransferSingleFilePieceBytes(String var1, long var2, byte[] var4) throws IOException;

    protected abstract byte[] getTransferMultiFilePieceBytes(String var1, long var2, byte[] var4) throws IOException;

    protected int nextSocketIndex() {
        return this.loadBalancer.nextSocketIndex();
    }

    protected boolean send(String pipeName, long creationTime, AirGapSocket socket, byte[] bytes) throws IOException {
        if (!socket.isConnected()) {
            throw new SocketException(String.format("Socket %s is closed, will try to handshake", socket));
        }
        bytes = this.compressIfNeeded(bytes);
        this.rateLimitIfNeeded(pipeName, creationTime, socket.getEndPoint(), bytes.length);
        BufferedOutputStream outputStream = new BufferedOutputStream(socket.getOutputStream());
        bytes = this.enrichWithLengthAndChecksum(bytes);
        outputStream.write(this.eLanguageEnable ? this.enrichWithELanguage(bytes) : bytes);
        outputStream.flush();
        byte[] response = new byte[1];
        int size = socket.getInputStream().read(response);
        return size > 0 && Arrays.equals(AirGapOneByteResponse.OK, response);
    }

    protected boolean send(AirGapSocket socket, byte[] bytes) throws IOException {
        return this.send(null, 0L, socket, bytes);
    }

    private byte[] enrichWithLengthAndChecksum(byte[] bytes) {
        byte[] length = BytesUtils.intToBytes((int)(bytes.length + 8));
        CRC32 crc32 = new CRC32();
        crc32.update(bytes, 0, bytes.length);
        return BytesUtils.concatByteArrayList(Arrays.asList(length, length, BytesUtils.longToBytes((long)crc32.getValue()), bytes));
    }

    private byte[] enrichWithELanguage(byte[] bytes) {
        return BytesUtils.concatByteArrayList(Arrays.asList(AirGapELanguageConstant.E_LANGUAGE_PREFIX, bytes, AirGapELanguageConstant.E_LANGUAGE_SUFFIX));
    }

    @Override
    public void close() {
        for (int i = 0; i < this.sockets.size(); ++i) {
            try {
                if (this.sockets.get(i) == null) continue;
                ((AirGapSocket)this.sockets.set(i, null)).close();
                continue;
            }
            catch (Exception e) {
                LOGGER.warn("Failed to close client {}.", (Object)i, (Object)e);
                continue;
            }
            finally {
                this.isSocketAlive.set(i, false);
            }
        }
        super.close();
    }

    private class RoundRobinLoadBalancer
    implements LoadBalancer {
        private RoundRobinLoadBalancer() {
        }

        @Override
        public int nextSocketIndex() {
            int socketSize = IoTDBAirGapConnector.this.sockets.size();
            for (int tryCount = 0; tryCount < socketSize; ++tryCount) {
                int clientIndex = (int)(IoTDBAirGapConnector.this.currentClientIndex++ % (long)socketSize);
                if (!Boolean.TRUE.equals(IoTDBAirGapConnector.this.isSocketAlive.get(clientIndex))) continue;
                return clientIndex;
            }
            throw new PipeConnectionException("All sockets are dead, please check the connection to the receiver.");
        }
    }

    private static interface LoadBalancer {
        public int nextSocketIndex();
    }

    private class RandomLoadBalancer
    implements LoadBalancer {
        private RandomLoadBalancer() {
        }

        @Override
        public int nextSocketIndex() {
            int socketSize = IoTDBAirGapConnector.this.sockets.size();
            int clientIndex = (int)(Math.random() * (double)socketSize);
            if (Boolean.TRUE.equals(IoTDBAirGapConnector.this.isSocketAlive.get(clientIndex))) {
                return clientIndex;
            }
            for (int tryCount = 0; tryCount < socketSize - 1; ++tryCount) {
                int nextClientIndex = (clientIndex + tryCount + 1) % socketSize;
                if (!Boolean.TRUE.equals(IoTDBAirGapConnector.this.isSocketAlive.get(nextClientIndex))) continue;
                return nextClientIndex;
            }
            throw new PipeConnectionException("All sockets are dead, please check the connection to the receiver.");
        }
    }

    private class PriorityLoadBalancer
    implements LoadBalancer {
        private PriorityLoadBalancer() {
        }

        @Override
        public int nextSocketIndex() {
            int socketSize = IoTDBAirGapConnector.this.sockets.size();
            for (int i = 0; i < socketSize; ++i) {
                if (!Boolean.TRUE.equals(IoTDBAirGapConnector.this.isSocketAlive.get(i))) continue;
                return i;
            }
            throw new PipeConnectionException("All sockets are dead, please check the connection to the receiver.");
        }
    }

    protected static class AirGapSocket
    extends Socket {
        private final TEndPoint endPoint;

        public AirGapSocket(String ip, int port) {
            this.endPoint = new TEndPoint(ip, port);
        }

        public TEndPoint getEndPoint() {
            return this.endPoint;
        }

        @Override
        public String toString() {
            return "AirGapSocket{endPoint=" + this.endPoint + "} (" + super.toString() + ")";
        }
    }
}

